unit SyntaxRichViewEdit;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  RVScroll, RichView, RVEdit, RVStyle, ExtCtrls;

type
  TSyntaxRichViewEdit = class(TRichViewEdit)
  private
    { Private declarations }
    FSyntaxUpdateTimer : TTimer;
    FTimerInterval : Cardinal;
    FSyntaxUpdating : boolean;
    FOnColorize: TNotifyEvent;
    procedure SetTimerInterval(Value: Cardinal);
  protected
    { Protected declarations }
    procedure DoUpdateSyntax(Sender: TObject);
  public
    { Public declarations }
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;

    procedure DoChange(ClearRedo: Boolean); override;
    procedure Undo; override;

    function ItemOffsToAbs(aItemNo, aOffs : integer) : integer;
    procedure AbsToItemOffs(aPos : integer; var aItemNo, aOffs : integer);
    procedure GetSelBounds(var aStart, aEnd : integer; var Inverted : boolean);
    procedure SetSelBounds(aStart, aEnd : integer; Inverted : boolean);
    function GetSelStart : integer;
    function GetSelEnd : integer;
    function GetSelLength : integer;
    procedure SetSelLength(aLen : integer);
    function GetAbsCaretPos : integer;
    procedure SetAbsCaretPos(aPos : integer);

    procedure Mark(aText : string; aStyleIndex : integer);
    procedure MarkBetween(aStartText, aEndText : string;
      aOuterStyleIndex, aInnerStyleIndex : integer;
      aAllowMultiline: boolean);
    procedure InsertTags(aTag1, aTag2 : string);
  published
    { Published declarations }
    property TimerInterval : Cardinal read FTimerInterval write SetTimerInterval default 100;
    property OnColorize: TNotifyEvent read FOnColorize write FOnColorize;
  end;

procedure Register;
function GetTextFromClipboard : string;

implementation

uses
  Clipbrd;

//------------------------------------------------------------------------------
procedure Register;
//------------------------------------------------------------------------------
begin
  RegisterComponents('RichView', [TSyntaxRichViewEdit]);
end;

//------------------------------------------------------------------------------
function GetTextFromClipboard : string;
//------------------------------------------------------------------------------
var
  Clipboard : TClipboard;
begin
  Clipboard := TClipboard.Create;
  Clipboard.Open;
  Result := Clipboard.AsText;
  Clipboard.Close;
  Clipboard.Free;
end;

//------------------------------------------------------------------------------
constructor TSyntaxRichViewEdit.Create(AOwner: TComponent);
//------------------------------------------------------------------------------
begin
  inherited Create(AOwner);
  Options := [rvoAllowSelection,rvoClientTextWidth,rvoShowPageBreaks,
              rvoAutoCopyText,rvoFormatInvalidate,rvoDblClickSelectsWord,
              rvoRClickDeselects];

  SetAddParagraphMode(False);

  FSyntaxUpdating := false;

  FTimerInterval := 100;

  FSyntaxUpdateTimer := TTimer.Create(nil);
  with FSyntaxUpdateTimer do
  begin
    Enabled := false;
    Interval := FTimerInterval;
    OnTimer := DoUpdateSyntax;
  end;
end;

//------------------------------------------------------------------------------
destructor TSyntaxRichViewEdit.Destroy;
//------------------------------------------------------------------------------
begin
  FSyntaxUpdateTimer.Free;
  inherited Destroy;
end;

//------------------------------------------------------------------------------
procedure TSyntaxRichViewEdit.DoChange(ClearRedo: Boolean);
//------------------------------------------------------------------------------
begin
  FSyntaxUpdateTimer.Enabled := false;
  if not FSyntaxUpdating then
  begin
    inherited DoChange(ClearRedo);

    if Assigned(FOnColorize) then // do nothing if no colorizer procedure defined
      FSyntaxUpdateTimer.Enabled := true;
  end;  
end;

//------------------------------------------------------------------------------
procedure TSyntaxRichViewEdit.Undo;
//------------------------------------------------------------------------------
begin
  BeginUpdate;
  if (UndoAction = rvutCustom) then
    inherited Undo;
  inherited Undo;
  EndUpdate;
end;

//------------------------------------------------------------------------------
procedure TSyntaxRichViewEdit.SetTimerInterval(Value: Cardinal);
//------------------------------------------------------------------------------
begin
  if (Value < 20) then Value := 20;
  if (FTimerInterval <> Value) then
  begin
    FTimerInterval := Value;
    FSyntaxUpdateTimer.Interval := FTimerInterval;
  end;
end;

//------------------------------------------------------------------------------
procedure TSyntaxRichViewEdit.DoUpdateSyntax(Sender: TObject);
//------------------------------------------------------------------------------
var
  OldSelStart,
  OldSelEnd : integer;
  OldInverted : boolean;
begin
  FSyntaxUpdateTimer.Enabled := false;
  FSyntaxUpdating := true;

  BeginUpdate;
  LockWindowUpdate(Handle);

  try
    GetSelBounds(OldSelStart, OldSelEnd, OldInverted);

    BeginUndoGroup(rvutCustom);
    SetUndoGroupMode(true);

    SelectAll;
    ApplyTextStyle(0);

    if (GetSelText <> '') then
    begin
      if Assigned(FOnColorize) then
        FOnColorize(Self);
    end;

    SetUndoGroupMode(false);

  finally
    LockWindowUpdate(0);
    EndUpdate;

    SetSelBounds(OldSelStart, OldSelEnd, OldInverted);

    FSyntaxUpdating := false;
  end;
end;

//------------------------------------------------------------------------------
function TSyntaxRichViewEdit.ItemOffsToAbs(aItemNo, aOffs : integer) : integer;
//------------------------------------------------------------------------------
var
  i, n : integer;
begin
  n := 0;
  for i := 0 to aItemNo-1 do
  begin
    if IsFromNewLine(i) then
      n := n + 1;

    if (GetItemStyle(i) >= 0) then // if it is a text item
      n := n + Length(GetItemText(i));
  end;
  n := n + aOffs-1;

  if IsFromNewLine(aItemNo) then
    n := n + 1;

  Result := n;
end;

//------------------------------------------------------------------------------
procedure TSyntaxRichViewEdit.AbsToItemOffs(aPos : integer;
  var aItemNo, aOffs : integer);
//------------------------------------------------------------------------------
var
  i, j, l, n : integer;
begin
  n := aPos;
  j := 0;
  for i := 0 to ItemCount-1 do
  begin
    j := i;

    l := 0;
    if (GetItemStyle(i) >= 0) then // if it is a text item
      l := Length(GetItemText(i));

    if IsFromNewLine(i) then
      l := l + 1;

    if n <= l then Break;

    n := n - l;
  end;

  if IsFromNewLine(j) then
    n := n - 1;

  aItemNo := j;
  aOffs := n + 1;
end;

//------------------------------------------------------------------------------
procedure TSyntaxRichViewEdit.GetSelBounds(var aStart, aEnd : integer;
  var Inverted : boolean);
//------------------------------------------------------------------------------
var
  aStartItem, aStartOffs,
  aEndItem, aEndOffs, i : integer;
begin
  RVData.GetSelectionBoundsEx(aStartItem, aStartOffs,
                              aEndItem, aEndOffs, false);

  aStart := ItemOffsToAbs(aStartItem, aStartOffs);
  aEnd := ItemOffsToAbs(aEndItem, aEndOffs);

  Inverted := (aStart > aEnd);
  if Inverted then
  begin // swap values
    i := aStart;
    aStart := aEnd;
    aEnd := i;
  end;
end;

//------------------------------------------------------------------------------
procedure TSyntaxRichViewEdit.SetSelBounds(aStart, aEnd : integer;
  Inverted : boolean);
//------------------------------------------------------------------------------
var
  aStartItem, aStartOffs,
  aEndItem, aEndOffs : integer;
begin
  AbsToItemOffs(aStart, aStartItem, aStartOffs);
  AbsToItemOffs(aEnd, aEndItem, aEndOffs);

  if Inverted then
    SetSelectionBounds(aEndItem, aEndOffs, aStartItem, aStartOffs)
  else
    SetSelectionBounds(aStartItem, aStartOffs, aEndItem, aEndOffs);
end;

//------------------------------------------------------------------------------
function TSyntaxRichViewEdit.GetSelStart : integer;
//------------------------------------------------------------------------------
var
  i : integer;
  f : boolean;
begin
  GetSelBounds(Result, i, f);
end;

//------------------------------------------------------------------------------
function TSyntaxRichViewEdit.GetSelEnd : integer;
//------------------------------------------------------------------------------
var
  i : integer;
  f : boolean;
begin
  GetSelBounds(i, Result, f);
end;

//------------------------------------------------------------------------------
function TSyntaxRichViewEdit.GetSelLength : integer;
//------------------------------------------------------------------------------
var
  aStart, aEnd : integer;
  f : boolean;
begin
  GetSelBounds(aStart, aEnd, f);
  Result := (aEnd - aStart);
end;

//------------------------------------------------------------------------------
procedure TSyntaxRichViewEdit.SetSelLength(aLen : integer);
//------------------------------------------------------------------------------
var
  aStartItem, aStartOffs,
  aEndItem, aEndOffs,
  aStart, aEnd : integer;
begin
  RVData.GetSelectionBoundsEx(aStartItem, aStartOffs,
                              aEndItem, aEndOffs, false);

  aStart := ItemOffsToAbs(aStartItem, aStartOffs);
  aEnd := ItemOffsToAbs(aEndItem, aEndOffs);

  if (aLen < 0) then aLen := 0;

  if (aEnd - aStart) = aLen then Exit;

  if (aLen > 0) then
    AbsToItemOffs(aStart + aLen, aEndItem, aEndOffs)
  else
  begin
    aEndItem := aStartItem;
    aEndOffs := aStartOffs;
  end;

  SetSelectionBounds(aStartItem, aStartOffs, aEndItem, aEndOffs);
end;

//------------------------------------------------------------------------------
function TSyntaxRichViewEdit.GetAbsCaretPos : integer;
//------------------------------------------------------------------------------
begin
  Result := ItemOffsToAbs(CurItemNo, OffsetInCurItem);
end;

//------------------------------------------------------------------------------
procedure TSyntaxRichViewEdit.SetAbsCaretPos(aPos : integer);
//------------------------------------------------------------------------------
var
  aItemNo, aOffs : integer;
begin
  AbsToItemOffs(aPos, aItemNo, aOffs);

  SetSelectionBounds(aItemNo, aOffs, aItemNo, aOffs);
end;

//------------------------------------------------------------------------------
procedure TSyntaxRichViewEdit.Mark(aText : string; aStyleIndex : integer);
//------------------------------------------------------------------------------
begin
  if ItemCount = 0 then Exit;
  SetSelectionBounds(0, GetOffsBeforeItem(0), 0, GetOffsBeforeItem(0));

  while SearchText(aText, [rvseoDown]) do
    ApplyTextStyle(aStyleIndex);
end;

//------------------------------------------------------------------------------
procedure TSyntaxRichViewEdit.MarkBetween(aStartText, aEndText : string;
  aOuterStyleIndex, aInnerStyleIndex : integer;
  aAllowMultiline: boolean);
//------------------------------------------------------------------------------
var
  StartItemNo, EndItemNo, i : integer;
  OldProtect : boolean;
begin
  if ItemCount = 0 then Exit;
  SetSelectionBounds(0, GetOffsBeforeItem(0), 0, GetOffsBeforeItem(0));

  while SearchText(aStartText, [rvseoDown]) do
  begin
    ApplyTextStyle(aOuterStyleIndex);

    StartItemNo := CurItemNo;

    if SearchText(aEndText, [rvseoDown]) then
    begin
      ApplyTextStyle(aOuterStyleIndex);

      EndItemNo := CurItemNo;

      If (StartItemNo < 1 + EndItemNo) then
      begin
        if not aAllowMultiline then
          for i := StartItemNo+1 to EndItemNo do
            if IsFromNewLine(i) then
              exit;
        OldProtect := (rvprConcateProtect in Style.TextStyles[aInnerStyleIndex].Protection);
        if not OldProtect then
          with Style.TextStyles[aInnerStyleIndex] do
            Protection := Protection + [rvprConcateProtect];

        for i := StartItemNo+1 to EndItemNo-1 do
          GetItem(i).StyleNo := aInnerStyleIndex;

        if not OldProtect then
          with Style.TextStyles[aInnerStyleIndex] do
            Protection := Protection - [rvprConcateProtect];
      end;
    end;
  end;
end;

//------------------------------------------------------------------------------
procedure TSyntaxRichViewEdit.InsertTags(aTag1, aTag2 : string);
//------------------------------------------------------------------------------
var
  OldSelStart, OldSelEnd, SelLen : integer;
  NewSelStart : integer;
  f : boolean;
begin
  BeginUpdate;
  LockWindowUpdate(Handle);
  try
    BeginUndoGroup(rvutInsert);
    SetUndoGroupMode(true);

    GetSelBounds(OldSelStart, OldSelEnd, f);
    SelLen := OldSelEnd - OldSelStart;

    SetAbsCaretPos(OldSelStart);
    InsertText(aTag1);

    NewSelStart := GetAbsCaretPos;

    SetAbsCaretPos(NewSelStart + SelLen);
    InsertText(aTag2);

    SetSelBounds(NewSelStart, NewSelStart + SelLen, f);

    SetUndoGroupMode(false);
  finally
    LockWindowUpdate(0);
    EndUpdate;
    //Change;
  end;
end;

end.
